package kr.co.joytune.dpush;

import kr.co.dpush.client.DPClient;
import kr.co.dpush.client.GroupInfo;
import kr.co.dpush.client.GroupOptions;
import kr.co.dpush.common.Callback;

import org.json.JSONObject;

import android.app.AlarmManager;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Handler;
import android.os.IBinder;
import android.os.PowerManager;
import android.os.SystemClock;
import android.support.v4.app.NotificationCompat;
import android.telephony.TelephonyManager;
import android.text.Html;
import android.util.Log;

public class DPushService extends Service {

	private static final String TAG = DPushService.class.getSimpleName();

	private Handler mHandler;

	private DPClient client;

	private BroadcastReceiver screenoff;

	private static final int NOTIFICATION_ID = 0x1234;
	private static final int ACTION_SHOW_NOTIFICATION = 0x5001;

	@Override
	public void onCreate() {

		// 등록된 알람은 제거
		Log.d(TAG, "onCreate()");
		unregisterRestartAlarm();

		super.onCreate();

		screenoff = new BroadcastReceiver() {

			@Override
			public void onReceive(Context context, Intent intent) {
				// 스마트폰 화면이 켜졌을때
				if (Intent.ACTION_SCREEN_ON.equals(intent.getAction())) {
					Log.d(TAG, "screen on..");
					WakeLocker.releaseWakeLock();
				}
				// 스마트폰 화면이 꺼졌을때
				else if (Intent.ACTION_SCREEN_OFF.equals(intent.getAction())) {
					Log.d(TAG, "screen off.");
					WakeLocker.acquireWakeLock(context);
				}
			}
		};

		IntentFilter offfilter = new IntentFilter(Intent.ACTION_SCREEN_OFF);
		offfilter.addAction(Intent.ACTION_SCREEN_ON);
		registerReceiver(screenoff, offfilter);

		// 5분후에 시작
		mHandler = new Handler() {

			public void handleMessage(android.os.Message msg) {
				switch (msg.what) {
				case ACTION_SHOW_NOTIFICATION:
					String message = (String) msg.obj;
					showNotification("오늘의 명언", message);
					break;
				case 0x8000:
					mHandler.removeMessages(0x8999);
					mHandler.sendEmptyMessageDelayed(0x8999, 1000);
					break;
				case 0x8999:
					// 서버에 conn
					if (!client.isConnected()) {
						connect();
						mHandler.sendEmptyMessageDelayed(0x8000, 10 * 1000);
					}
					break;
				}
			};
		};
	}

	@Override
	public IBinder onBind(Intent intent) {
		return null;
	}

	@Override
	public int onStartCommand(Intent intent, int flags, int startId) {
		Log.d(TAG, "onStartCommand()");

		if (client == null || !client.isConnected()) {
			connect();
		}

		return super.onStartCommand(intent, flags, startId);
	}

	@Override
	public void onDestroy() {
		Log.d(TAG, "onDestroy()");
		// 알람 등록
		registerRestartAlarm();
		// Screen Off Receiver 를 해제한다
		unregisterReceiver(screenoff);

		super.onDestroy();
	}

	/**
	 * 발급받은 product key
	 * TODO 키 정의
	 */
	private String PRD_KEY = null;
	/**
	 * 생성하고자 하는 그룹명
	 * TODO Group 정의
	 */
	private String GRP_ID = null;
	/**
	 * 메시지 작업단위
	 * TODO Action 정의
	 */
	private String ACT_ID = null;

	private GroupInfo gi;

	private void connect() {

		if (PRD_KEY == null || "".equals(PRD_KEY)) {
			Log.e(TAG, "productkey 가 정의되어있지 않습니다.");
			return;
		}

		if (GRP_ID == null || "".equals(GRP_ID)) {
			Log.e(TAG, "groupId 가 정의되어있지 않습니다.");
			return;
		}

		if (ACT_ID == null || "".equals(ACT_ID)) {
			Log.e(TAG, "actionId 가 정의되어있지 않습니다.");
			return;
		}

		// 기 정의된 dpclient 가 없으면 생성한다
		if (this.client == null) {
			this.client = new DPClient(PRD_KEY) {

				public void onConnected() {
					Log.d(TAG, "onconnect................");
				}

				public void onDisconnected() {
					Log.d(TAG, "ondisconnect................");
					// 종료시 재접속 요청(30초후)
					mHandler.sendEmptyMessageDelayed(0x8999, 30 * 1000);
				}
			};
		}

		// 서버와의 접속을 시도한다
		if (!client.isConnected()) {
			client.connect();
		}

		// 오픈된 그룹이면 정의된 ActionID 에 해당하는 callback만을 재정의한다
		if (client.isOpened(GRP_ID)) {
			Log.d(TAG, "is opened....");
			gi = client.getOpenedGroup(GRP_ID);
			gi.onReceive(ACT_ID, action(mHandler));
		}
		// 그룹을 오픈한다
		else {
			Log.d(TAG, "is not opened....");

			GroupOptions gpopts = new GroupOptions();
			// - 그룹에서 CUSTMER INFO 이벤트를 callback 받는 유무, 기본값 false
			// gpopts.setCustevent(true);
			// - 해당 그룹의 client가 send 함수를 사용가능한 유무. 기본값 false
			// gpopts.setSendevent(true);

			JSONObject cinfo = new JSONObject();
			try {
				//
				TelephonyManager mTelephonyManager = (TelephonyManager) getApplicationContext().getSystemService(Context.TELEPHONY_SERVICE);
				String telephone = mTelephonyManager.getLine1Number();
				cinfo.put("nickname", telephone);
				cinfo.put("gender", "P");
				cinfo.put("age", "0");
			}
			catch (Exception e) {
				e.printStackTrace();
			}

			gpopts.setCustinfo(cinfo);

			gi = client.openGroup(GRP_ID, gpopts, null);
			gi.onReceive(ACT_ID, action(mHandler));
		}
	}

	/**
	 * 서버에서 메시지가 수신되었을때의 callback을 정의한다
	 * @param handler
	 * @return
	 */
	private static Callback action(final Handler handler) {
		return new Callback() {
			@Override
			public void call(Object... args) {
				String message = (String) args[0];
				if (message != null && !"".equals(message)) {
					handler.sendMessage(handler.obtainMessage(ACTION_SHOW_NOTIFICATION, message));
				}
			}
		};
	}

	/**
	 * 알림영역에 Notification 을 노출한다
	 * @param title
	 * @param text
	 */
	private void showNotification(CharSequence title, CharSequence text) {

		// 알림 도착시 실행하고자 하는 작업(Intent)을 설정
		Intent intent = new Intent(DPushService.this.getApplicationContext(), DPushAlarm.class);
		// - 필요한 parameter 설정
		intent.putExtra("NM", title.toString());
		intent.putExtra("MG", text.toString());
		// - 기존에 호출된 Intent 가 있으면 제거한다
		intent.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
		// - 새로운 Activity를 생성하도록 한다
		intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

		// PendingIntent 초기화
		PendingIntent contentIntent = PendingIntent.getActivity(getApplicationContext(), 0, intent, PendingIntent.FLAG_UPDATE_CURRENT);

		// 메시지 도착을 알리기 위해 Notification을 사용한다
		// Notification Builder 객체생성
		NotificationCompat.Builder notibuilder = new NotificationCompat.Builder(getApplicationContext());
		// - 상태표시줄에 나타낼 아이콘
		notibuilder.setSmallIcon(R.drawable.dpush_logo);
		// - Notification 을 표시할 시각
		notibuilder.setWhen(System.currentTimeMillis());
		// - 상태표시줄에 나타낼 제목
		notibuilder.setContentTitle(title);
		// - 상태표시줄에 나타낼 내용
		notibuilder.setContentText(Html.fromHtml(text.toString()));
		// - 상태바에 알림도착시 나타낼 텍스트(Ticker)
		// notibuilder.setTicker(Html.fromHtml(text.toString()));
		// - 알림항목을 눌렀을때 해당 알림을 자동으로 해제시키도록 설정
		notibuilder.setAutoCancel(true);
		// - 알림항목을 눌렀을때 실행할 작업(Intent)
		notibuilder.setContentIntent(contentIntent);
		// - builder 를 통해 Notification 생성
		Notification notification = notibuilder.build();

		// - 알림도착시 소리
		notification.defaults |= Notification.DEFAULT_SOUND;

		// NotificationManager 객체 호출
		NotificationManager mNM = (NotificationManager) getSystemService(NOTIFICATION_SERVICE);
		// - 고유아이디를가지는 notification 을 표시
		mNM.notify(NOTIFICATION_ID, notification);
	}

	/**
	 * 서비스가 시스템에 의해서 또는 강제적으로 종료되었을 때 호출되어
	 * 알람을 등록해서 15초 후에 서비스가 실행되도록 한다.
	 */
	private void registerRestartAlarm() {

		Log.d(TAG, "registerRestartAlarm()");

		Intent intent = new Intent(DPushService.this, DPushReceiver.class);
		intent.setAction(DPushReceiver.ACTION_DPUSH_RESTART);
		PendingIntent sender = PendingIntent.getBroadcast(DPushService.this, 0, intent, 0);

		long firstTime = SystemClock.elapsedRealtime();
		firstTime += 15 * 1000; // 15초 후에 알람이벤트 발생

		AlarmManager am = (AlarmManager) getSystemService(ALARM_SERVICE);
		am.setRepeating(AlarmManager.ELAPSED_REALTIME_WAKEUP, firstTime, 60 * 1000, sender);
	}

	/**
	 * 기존 등록되어있는 알람을 해제한다.
	 */
	private void unregisterRestartAlarm() {

		Log.d(TAG, "unregisterRestartAlarm()");
		Intent intent = new Intent(DPushService.this, DPushReceiver.class);
		intent.setAction(DPushReceiver.ACTION_DPUSH_RESTART);
		PendingIntent sender = PendingIntent.getBroadcast(DPushService.this, 0, intent, 0);

		AlarmManager am = (AlarmManager) getSystemService(ALARM_SERVICE);
		am.cancel(sender);
	}

	static class WakeLocker {

		private static PowerManager.WakeLock sCpuWakeLock;

		/**
		 * PowerManager.WakeLock 시작
		 * @param context
		 */
		static void acquireWakeLock(Context context) {
			Log.d(TAG, "Acquiring cpu wake lock");
			PowerManager pm = (PowerManager) context.getSystemService(Context.POWER_SERVICE);
			// 참고 http://aroundck.tistory.com/48
			if (!pm.isScreenOn()) {
				sCpuWakeLock = pm.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, context.getClass().getSimpleName());
				sCpuWakeLock.acquire(2 * 60 * 1000);
			}
		}

		/**
		 * PowerManager.WakeLock 해제
		 */
		static void releaseWakeLock() {
			Log.d(TAG, "Releasing cpu wake lock");
			if (sCpuWakeLock != null && sCpuWakeLock.isHeld()) {
				sCpuWakeLock.release();
			}
		}
	}
}