//
//  DPush.h
//  DPush
//
//  Created by 조영광 on 2015. 3. 3..
//  Copyright (c) 2015 Joytune. All rights reserved.
//

#ifndef DPush_DPClient_h
#define DPush_DPClient_h

#import <Foundation/Foundation.h>
#import <JavaScriptCore/JavaScriptCore.h>

// =========================================================================================
/**
 Group 정보 Class
 */
@interface GroupInfo : NSObject
// generator
+ (GroupInfo*) initWithId:(NSString*)groupid;
+ (GroupInfo*) initWithId:(NSString*)groupid custevent:(BOOL)iscustevent sendevent:(BOOL)issendevent;
// Event responders
/**
 @brief openGroup 시의 callback
 */
@property (nonatomic, copy) void (^onCallback)();
/**
 @brief 유저가 그룹에 조인한경우
 @arg cid custid
 @arg custinfo 유저정보
 */
@property (nonatomic, copy) void (^onUserIn)(NSString* cid, NSObject* custinfo);
/**
 @brief 유저가 그룹에서 탈퇴한경우
 @arg cid custid
 @arg custinfo 유저정보
 */
@property (nonatomic, copy) void (^onUserOut)(NSString* cid, NSObject* custinfo);
/**
 @brief 유저 정보가 변경된 경우
 @arg cid custid
 @arg custinfo 유저정보
 */
@property (nonatomic, copy) void (^onUserUpdated)(NSString* cid, NSObject* custinfo);
@property (nonatomic, copy) void (^onAuth)();
// Method
/**
 @brief Push된 메시지 중에서 action과 동일한 이름이 있으면 callback함수로 전달한다
 @param action 임의의 action명
 @param callback 메시지를 전달 받는 callback함수
 */
- (void)onReceive:(NSString *)action callback:(void (^)(NSObject *args))function;
/**
 @brief action명으로 message를 전체 Client에게 전달한다
 @param action 임의의 action명
 @param message 전달할 데이터의 내용
 @remark openGroup시 sendevent
 */
- (void)send:(NSString *)action message:(NSString *)message;
/**
 @brief Open된 그룹에 접속되어 있는 user의 목록을 반환한다
 @return user목록
 @remark custevent:true 인 경우에 한함
 */
- (NSDictionary*)getUserList;
/**
 @brief Open된 그룹에 접속되어 있는 user중 특정 custid를 가지고 있는 user정보를 반환한다
 @param cid custid
 @return user정보
 @remark custevent:true 인 경우에 한함
 */
- (NSObject*)getUserInfo:(NSString*)cid;
/**
 @brief Client의 사용자 정보가 변경되었을 경우 전체 Client에게 변경된 정보를 전달
 [group updateUser:@{@"nickname":@"홍길동"}]
 @param custinfo 변경된 사용자 정보
 */
- (void)updateUser:(NSDictionary *)custinfo;
@end

// =========================================================================================
/**
 Group Option 정보 Class
 */
@interface GroupOptions : NSObject
/**
 @brief custevent 사용여부
 */
@property BOOL custevent;
/**
 @brief 메시지 송신 가능여부
 */
@property BOOL sendevent;
/**
 @brief 사용자정보
 */
@property NSDictionary* custinfo;
/**
 @brief 인증파라메터
 */
@property NSString* authparam;
// Event responders
/**
 @brief 인증 callback
 */
@property (nonatomic, copy) void (^onAuth)();

@end

// =========================================================================================
/**
 @brief DPClient 용 옵션 클래스
 */
@interface DPOptions : NSObject
/**
 @brief SSL 사용여부
 */
@property BOOL security;
/**
 @brief 인증파라메터
 */
@property NSString* authparam;
/**
 @brief 재접속 시도 횟수
 */
@property NSNumber* reconnectionAttempts;
/**
 @brief 재접속 시도 delay
 */
@property NSNumber* reconnectionDelay;
/**
 @brief 재접속 시도 delay 최대값
 */
@property NSNumber* reconnectionDelayMax;
// Event responders
/**
 * callback
 * @param rc 응답코드
 * @param msg 메시지
 */
@property (nonatomic, copy) void (^onCallback)(NSInteger rc, NSString* msg);
@end

// =========================================================================================
/**
 DPClient
 @author joytune http://www.joytune.co.kr
 */
@interface DPClient : NSObject {
    
}

// Generators
/**
 생성자
 @param productKey  가입시 발급받은 인증키
 */
+ (DPClient *)key:(NSString *)productKey;
/**
 생성자
 @param productKey 가입시 발급받은 인증키
 @param options 옵션정보
 */
+ (DPClient *)key:(NSString *)productKey options:(DPOptions *)options;

@property BOOL isLogging;
// Event responders
@property (nonatomic, copy) void (^onConnect)();
@property (nonatomic, copy) void (^onDisconnect)();

- (void)connect;
- (void)close;

/**
 @brief     현재 접속된 client의 고유한 id(userid)를 반환한다
 @return    client 고유 id(userid)
 */
- (NSString *)getUserId;
/**
 @brief 특정 그룹으로 접속한다. 최초 접속시 생성
 @param groupid 임의의 그룹명
 @return 그룹정보
 */
- (GroupInfo *)openGroup:(NSString *)groupid;
/**
 @brief 특정 그룹으로 접속한다. 최초 접속시 생성
 @param groupid 임의의 그룹명
 @param options 그룹의 옵션정보
 @return 그룹정보
 */
- (GroupInfo *)openGroup:(NSString *)groupid options:(GroupOptions*)options;
/**
 @brief 특정 그룹으로 접속한다. 최초 접속시 생성
 @param groupid 임의의 그룹명
 @param options 그룹의 옵션정보
 @param callback 그룹이 생성된 후의 이벤트 callback
 @return 그룹정보
 */
- (GroupInfo *)openGroup:(NSString *)groupid options:(GroupOptions*)options callback:(void(^)())callback;
/**
 @brief Open되어있는 그룹정보 반환
 @param groupid 임의의 그룹명
 @return 그룹정보
 */
- (GroupInfo *)getOpenedGroup:(NSString *)groupid;

@end

#endif
