<?php
/**
 * @fileoverview dpush php server API
 * @author ben
 * @version 0.5
 * @since 2015.03.19
 */

/**
 * @todo
 */
abstract class AbstractSocketIO implements EngineInterface
{
	const CONNECT      = 0;
	const DISCONNECT   = 1;
	const EVENT        = 2;
	const ACK          = 3;
	const ERROR        = 4;
	const BINARY_EVENT = 5;
	const BINARY_ACK   = 6;

	/** @var string[] Parse url result */
	protected $url;

	/** @var string[] Session information */
	protected $session;

	/** @var mixed[] Array of options for the engine */
	protected $options;

	/** @var resource Resource to the connected stream */
	protected $stream;

	/** @var nsp Namespace */
	protected $nsp = '/';

	/*
	 public function __construct($url, array $options = [])
	 {
	 $this->url     = $this->parseUrl($url);
	 $this->options = array_replace($this->getDefaultOptions(), $options);
	 }
	 */

	public function __construct($url, $nsp = '/', array $options = array()	)
	{
		$this->url     = $this->parseUrl($url);
		$this->options = array_replace($this->getDefaultOptions(), $options);
		$this->nsp = $nsp;
	}

	/** {@inheritDoc} */
	public function connect()
	{
		throw new UnsupportedActionException($this, 'connect');
	}

	/** {@inheritDoc} */
	public function keepAlive()
	{
		throw new UnsupportedActionException($this, 'keepAlive');
	}

	/** {@inheritDoc} */
	public function close()
	{
		throw new UnsupportedActionException($this, 'close');
	}

	/**
	 * Write the message to the socket
	 *
	 * @param integer $code    type of message (one of EngineInterface constants)
	 * @param string  $message Message to send, correctly formatted
	 */
	abstract public function write($code, $message = null);

	/** {@inheritDoc} */
	public function emit($event, array $args)
	{
		throw new UnsupportedActionException($this, 'emit');
	}

	/**
	 * {@inheritDoc}
	 *
	 * Be careful, this method may hang your script, as we're not in a non
	 * blocking mode.
	 */
	public function read() {
		// Ignore first byte, I hope Socket.io does not send fragmented frames, so we don't have to deal with FIN bit.
		// There are also reserved bit's which are 0 in socket.io, and opcode, which is always "text frame" in Socket.io
		$header  = fread($this->stream, 1);
		// There is also masking bit, as MSB, but it's 0 in current Socket.io
		$payload_len = ord(fread($this->stream, 1));

		switch ($payload_len) {
			case 126:
				$payload_len = unpack("n", fread($this->stream, 2));
				$payload_len = $payload_len[1];
				break;
			case 127:
				break;
		}
		// Reads the socket until full data is read
		//$payload = fread($this->fd, $payload_len); // Does not works if packet size > 16Kb
		$payload = '';
		$read    = 0;
		while( $read < $payload_len && ( $buf = fread( $this->stream, $payload_len - $read ) ) )
		{
			$read    += strlen($buf);
			$payload .= $buf;
		}

		// decode the payload
		return (string) new Decoder($payload);
	}


	/** {@inheritDoc} */
	public function getName()
	{
		return 'SocketIO';
	}

	/**
	 * Parse an url into parts we may expect
	 *
	 * @return string[] information on the given URL
	 */
	protected function parseUrl($url)
	{
		$parsed = parse_url($url);

		if (false === $parsed) {
			throw new MalformedUrlException($url);
		}

		$server = array_replace(array('scheme' => 'http',
				'host'   => 'localhost',
				'query'  => array(),
				'path'   => 'socket.io'), $parsed);

		if (!isset($server['port'])) {
			$server['port'] = 'https' === $server['scheme'] ? 443 : 80;
		}

		if (!is_array($server['query'])) {
			parse_str($server['query'], $query);
			$server['query'] = $query;
		}

		$server['secured'] = 'https' === $server['scheme'];

		return $server;
	}

	/**
	 * Get the defaults options
	 *
	 * @return array mixed[] Defaults options for this engine
	 */
	protected function getDefaultOptions()
	{
		return array('context' => array(),
				'debug'   => false,
				'wait'    => 100*1000,
				'timeout' => ini_get("default_socket_timeout")
				//'timeout' => "1"
		);
	}
}

/**
 * @ignore
 */
class Version1X extends AbstractSocketIO
{
	const TRANSPORT_POLLING   = 'polling';
	const TRANSPORT_WEBSOCKET = 'websocket';

	/** {@inheritDoc} */
	public function connect()
	{
		if (is_resource($this->stream)) {
			return;
		}

		$this->handshake();

		$errors = array(null, null);
		$host   = sprintf('%s:%d', $this->url['host'], $this->url['port']);

		if (true === $this->url['secured']) {
			$host = 'ssl://' . $host;
		}

		$this->stream = stream_socket_client($host, $errors[0], $errors[1], $this->options['timeout'], STREAM_CLIENT_CONNECT, stream_context_create($this->options['context']));

		if (!is_resource($this->stream)) {
			throw new SocketException($error[0], $error[1]);
		}
		stream_set_timeout($this->stream, $this->options['timeout']);
		$this->upgradeTransport();
		$this->connectToNamespace();


	}

	/** {@inheritDoc} */
	public function close()
	{

		if (!is_resource($this->stream)) {
			return;
		}


		$this->write(EngineInterface::CLOSE);

		fclose($this->stream);
		$this->stream = null;
	}

	/** {@inheritDoc} */
	public function emit($event, array $args)
	{
		//var_dump(EngineInterface::MESSAGE, static::EVENT . "/" . $this->nsp . ',' . json_encode([$event, $args]));
		//return $this->write(EngineInterface::MESSAGE, static::EVENT . json_encode([$event, $args]));
		return $this->write(EngineInterface::MESSAGE, static::EVENT . "/" . $this->nsp . ',' . json_encode(array($event, $args)));
	}

	/** {@inheritDoc} */
	public function write($code, $message = null)
	{
		if (!is_resource($this->stream)) {
			return;
		}

		if (!is_int($code) || 0 > $code || 6 < $code) {
			throw new InvalidArgumentException('Wrong message type when trying to write on the socket');
		}

		$payload = new Encoder($code . $message, Encoder::OPCODE_TEXT, true);
		/*
		 echo "<br>";
		 var_dump($code . $message);
		 echo "<br>";
		*/
		$bytes = fwrite($this->stream, (string) $payload);

		// wait a little bit of time after this message was sent
		usleep((int) $this->options['wait']);


		return $bytes;
	}

	/** {@inheritDoc} */
	public function getName()
	{
		return 'SocketIO Version 1.X';
	}

	/** {@inheritDoc} */
	protected function getDefaultOptions()
	{
		$defaults = parent::getDefaultOptions();

		$defaults['version']   = 3;
		$defaults['use_b64']   = false;
		$defaults['transport'] = static::TRANSPORT_POLLING;
		$defaults['cookies']   = array();

		return $defaults;
	}

	protected function connectToNamespace()
	{
		if (!is_string($this->nsp) || $this->nsp == '/')
			return;
		$this->write(EngineInterface::MESSAGE, static::CONNECT . "/" . $this->nsp);

		$this->read();		// 여기서 한번 READ 해준다. 해당 write 된 부분에 대한 read
	}

	/** Does the handshake with the Socket.io server and populates the `session` value object */
	protected function handshake()
	{
		if (null !== $this->session) {
			return;
		}

		$query = array('use_b64'   => $this->options['use_b64'],
				'EIO'       => $this->options['version'],
				'transport' => $this->options['transport'],
				'PRODUCTKEY' => $this->options['productkey']
		);

		if (isset($this->url['query'])) {
			$query = array_replace($query, $this->url['query']);
		}

		// 20150324 mistyi add = PHP3.3 에서의 오류로 아래 1라인 수정함. 
		// EX) sid=mxa9tOu1PnYV-c7rAAWS&amp;EIO=3&amp;use_b64=0&amp;transport=websocket 와 같이 파라미터가 &amp; 로 연결됨
		//$url    = sprintf('%s://%s:%d/%s/?%s&CONTYPE=server', $this->url['scheme'], $this->url['host'], $this->url['port'], trim($this->url['path'], '/'), http_build_query($query));
		$url    = sprintf('%s://%s:%d/%s/?%s&CONTYPE=server', $this->url['scheme'], $this->url['host'], $this->url['port'], trim($this->url['path'], '/'), http_build_query($query,'','&'));

		$result = @file_get_contents($url, false, stream_context_create(array('http' => array('timeout' => (float) $this->options['timeout']))));

		// 최초 접속시에 cookie 값을 그대로 저장해야 한다. 2015-01-19 dhwogh
		$cookies = array();
		foreach ($http_response_header as $hdr) {
			if (preg_match('/^Set-Cookie:\s*([^;]+)/', $hdr, $matches)) {
				parse_str($matches[1], $tmp);
				$cookies += $tmp;
			}
		}
		$this->options['cookies'] = $cookies;

		if (false === $result) {
			throw new ServerConnectionFailureException;
		}

		$decoded = json_decode(substr($result, strpos($result, '{')), true);

		if (!in_array('websocket', $decoded['upgrades'])) {
			throw new UnsupportedTransportException('websocket');
		}

		$this->session = new Session($decoded['sid'], $decoded['pingInterval'], $decoded['pingTimeout'], $decoded['upgrades']);

	}

	/** Upgrades the transport to WebSocket */
	private function upgradeTransport()
	{
		$query = array('sid'       => $this->session->id,
				'EIO'       => $this->options['version'],
				'use_b64'   => $this->options['use_b64'],
				'transport' => static::TRANSPORT_WEBSOCKET);

		// 20150324 mistyi add = PHP3.3 에서의 오류로 아래 1라인 수정함. 
		// EX) sid=mxa9tOu1PnYV-c7rAAWS&amp;EIO=3&amp;use_b64=0&amp;transport=websocket 와 같이 파라미터가 &amp; 로 연결됨
		//$url = sprintf('/%s/?%s', trim($this->url['path'], '/'), http_build_query($query));
		$url = sprintf('/%s/?%s', trim($this->url['path'], '/'), http_build_query($query,'','&'));
		
		$key = base64_encode(sha1(uniqid(mt_rand(), true), true));

		// 웹소켓 upgrade 시에 해당 cookie값을 전달해야 haproxy 설정이 제대로 작동한다. 2015-01-19 dhwogh
		$cookiestr = "";
		$cookiearr = $this->options['cookies'];
		if(!empty($cookiearr)) {
			$cookiestr = "Cookie: ";
			foreach ($cookiearr as $name => $value) {
				$cookiestr .= $name . "=" . $value .";";
			}
		}

		$request = "GET {$url} HTTP/1.1\r\n";
		$request .= $cookiestr;
		$request .= "Host: {$this->url['host']}\r\n"
		. "Upgrade: WebSocket\r\n"
				. "Connection: Upgrade\r\n"
						. "Sec-WebSocket-Key: {$key}\r\n"
						. "Sec-WebSocket-Version: 13\r\n"
								. "Origin: *\r\n\r\n";

		fwrite($this->stream, $request);
		$result = fread($this->stream, 12);

		if ('HTTP/1.1 101' !== $result) {
			throw new UnexpectedValueException(sprintf('The server returned an unexpected value. Expected "HTTP/1.1 101", had "%s"', $result));
		}

		// cleaning up the stream
		while ('' !== trim(fgets($this->stream)));

		$this->write(EngineInterface::UPGRADE);

		$this->read();		// 여기서 한번 READ 해준다. 해당 write 된 부분에 대한 read
	}
}

/**
 * @ignore
 */
class Session
{
	/** @var integer session's id */
	private $id;

	/** @var integer session's last heartbeat */
	private $heartbeat;

	/** @var integer[] session's and heartbeat's timeouts */
	private $timeouts;

	/** @var string[] supported upgrades */
	private $upgrades;

	public function __construct($id, $interval, $timeout, array $upgrades)
	{
		$this->id        = $id;
		$this->upgrades  = $upgrades;
		$this->heartbeat = time();

		$this->timeouts  = array('timeout'  => $timeout,
				'interval' => $interval);
	}

	/** The property should not be modified, hence the private accessibility on them */
	public function __get($prop)
	{
		static $list = array('id', 'upgrades');

		if (!in_array($prop, $list)) {
			throw new InvalidArgumentException(sprintf('Unknown property "%s" for the Session object. Only the following are availables : ["%s"]', $prop, implode('", "', $list)));
		}

		return $this->$prop;
	}

	/**
	 * Checks whether a new heartbeat is necessary, and does a new heartbeat if it is the case
	 *
	 * @return Boolean true if there was a heartbeat, false otherwise
	 */
	public function needsHeartbeat()
	{
		if (0 < $this->timeouts['interval'] && time() > ($this->timeouts['interval'] + $this->heartbeat - 5)) {
			$this->heartbeat = time();

			return true;
		}

		return false;
	}
}

/**
 * @ignore
 */
class Client
{
	/** @var EngineInterface */
	private $engine;
	/** @var LoggerInterface */
	private $logger;
	private $isConnected = false;
	public function __construct(EngineInterface $engine, LoggerInterface $logger = null)
	{
		$this->engine = $engine;
		$this->logger = $logger;
	}
	public function __destruct()
	{
		if (!$this->isConnected) {
			return;
		}
		$this->close();
	}
	/**
	 * Connects to the websocket
	 *
	 * @param boolean $keepAlive keep alive the connection (not supported yet) ?
	 * @return $this
	 */
	public function initialize($keepAlive = false)
	{
		try {
			null !== $this->logger && $this->logger->debug('Connecting to the websocket');
			$this->engine->connect();
			null !== $this->logger && $this->logger->debug('Connected to the server');
			$this->isConnected = true;
			if (true === $keepAlive) {
				null !== $this->logger && $this->logger->debug('Keeping alive the connection to the websocket');
				$this->engine->keepAlive();
			}
		} catch (SocketException $e) {
			null !== $this->logger && $this->logger->error('Could not connect to the server', array('exception' => $e));
			throw $e;
		}
		return $this;
	}
	/**
	 * Reads a message from the socket
	 *
	 * @return MessageInterface Message read from the socket
	 */
	public function read()
	{
		null !== $this->logger && $this->logger->debug('Reading a new message from the socket');
		return $this->engine->read();
	}
	/**
	 * Emits a message through the engine
	 *
	 * @return $this
	 */
	public function emit($event, array $args)
	{
		null !== $this->logger && $this->logger->debug('Sending a new message', array('event' => $event, 'args' => $args));
		$this->engine->emit($event, $args);
		return $this;
	}
	/**
	 * Closes the connection
	 *
	 * @return $this
	 */
	public function close()
	{
		null !== $this->logger && $this->logger->debug('Closing the connection to the websocket');
		$this->engine->close();
		$this->isConnected = false;
		return $this;
	}
	/**
	 * Gets the engine used, for more advanced functions
	 *
	 * @return EngineInterface
	 */
	public function getEngine()
	{
		return $this->engine;
	}
}




/**
 * @ignore
 */
interface EngineInterface
{
	const OPEN    = 0;
	const CLOSE   = 1;
	const PING    = 2;
	const PONG    = 3;
	const MESSAGE = 4;
	const UPGRADE = 5;
	const NOOP    = 6;

	/** Connect to the targeted server */
	public function connect();

	/** Closes the connection to the websocket */
	public function close();

	/**
	 * Read data from the socket
	 *
	 * @return string Data read from the socket
	*/
	public function read();

	/**
	 * Emits a message through the websocket
	 *
	 * @param string $event Event to emit
	 * @param array  $args  Arguments to send
	*/
	public function emit($event, array $args);

	/** Keeps alive the connection */
	public function keepAlive();

	/** Gets the name of the engine */
	public function getName();
}

/**
 * @ignore
 */
class Encoder extends AbstractPayload
{
	private $data;
	private $payload;

	/**
	 * @param string  $data   data to encode
	 * @param integer $opcode OpCode to use (one of AbstractPayload's constant)
	 * @param bool    $mask   Should we use a mask ?
	 */
	public function __construct($data, $opCode, $mask)
	{
		$this->data    = $data;
		$this->opCode  = $opCode;
		$this->mask    = (bool) $mask;

		if (true === $this->mask) {
			$this->maskKey = openssl_random_pseudo_bytes(4);
		}
	}

	public function encode()
	{
		if (null !== $this->payload) {
			return;
		}

		$pack   = '';
		$length = strlen($this->data);

		if (0xFFFF < $length) {
			//$pack   = pack('NN', ($length & 0xFFFFFFFF00000000) >> 0b100000, $length & 0x00000000FFFFFFFF);
			$pack   = pack('NN', ($length & 0xFFFFFFFF00000000) >> bindec('100000'), $length & 0x00000000FFFFFFFF);
			$length = 0x007F;
		} elseif (0x007D < $length) {
			$pack   = pack('n*', $length);
			$length = 0x007E;
		}

		$payload = ($this->fin << bindec('001')) | $this->rsv[0];
		$payload = ($payload   << bindec('001')) | $this->rsv[1];
		$payload = ($payload   << bindec('001')) | $this->rsv[2];
		$payload = ($payload   << bindec('100')) | $this->opCode;
		$payload = ($payload   << bindec('001')) | $this->mask;
		$payload = ($payload   << bindec('111')) | $length;

		$data    = $this->data;
		$payload = pack('n', $payload) . $pack;

		if (true === $this->mask) {
			$payload .= $this->maskKey;
			$data     = $this->maskData($data);
		}

		$this->payload = $payload . $data;
	}

	public function __toString()
	{
		$this->encode();

		return $this->payload;
	}
}

/**
 * @ignore
 */
class Decoder extends AbstractPayload implements Countable
{
	private $payload;
	private $data;

	private $length;

	/** @param string $payload Payload to decode */
	public function __construct($payload)
	{
		$this->payload = $payload;
	}

	public function decode()
	{
		if (null !== $this->data) {
			return;
		}

		$length = count($this);



		// if ($payload !== null) and ($payload packet error)?
		// invalid websocket packet data or not (text, binary opCode)
		if (3 > $length) {
			return;
		}

		$payload = array_map('ord', str_split($this->payload));

		$this->fin = ($payload[0] >> bindec('111'));

		$this->rsv = array(($payload[0] >> bindec('110')) & bindec('1'),  // rsv1
				($payload[0] >> bindec('101')) & bindec('1'),  // rsv2
				($payload[0] >> bindec('100')) & bindec('1')); // rsv3

		$this->opCode = $payload[0] & 0xF;
		$this->mask   = (bool) ($payload[1] >> bindec('111'));

		$payloadOffset = 2;

		if ($length > 125) {
			$payloadOffset = (0xFFFF < $length && 0xFFFFFFFF >= $length) ? 6 : 4;
		}

		$payload = implode('', array_map('chr', $payload));

		if (true === $this->mask) {
			$this->maskKey  = substr($payload, $payloadOffset, 4);
			$payloadOffset += 4;
		}

		$data = substr($payload, $payloadOffset, $length);

		if (true === $this->mask) {
			$data = $this->maskData($data);
		}

		$this->data = $data;
	}

	public function count()
	{
		if (null === $this->payload) {
			return 0;
		}

		if (null !== $this->length) {
			return $this->length;
		}

		if(gettype($this->payload) == "string") {
			return strlen($this->payload);
		}

		$length = ord($this->payload[1]) & 0x7F;

		if ($length == 126 || $length == 127) {
			$length = unpack('H*', substr($this->payload, 2, ($length == 126 ? 2 : 4)));
			$length = hexdec($length[1]);
		}

		return $this->length = $length;
	}

	public function __toString()
	{
		$this->decode();

		return $this->data ?: '';
	}
}

/**
 * @ignore
 */
abstract class AbstractPayload
{
	const OPCODE_NON_CONTROL_RESERVED_1 = 0x3;
	const OPCODE_NON_CONTROL_RESERVED_2 = 0x4;
	const OPCODE_NON_CONTROL_RESERVED_3 = 0x5;
	const OPCODE_NON_CONTROL_RESERVED_4 = 0x6;
	const OPCODE_NON_CONTROL_RESERVED_5 = 0x7;

	const OPCODE_CONTINUE = 0x0;
	const OPCODE_TEXT     = 0x1;
	const OPCODE_BINARY   = 0x2;
	const OPCODE_CLOSE    = 0x8;
	const OPCODE_PING     = 0x9;
	const OPCODE_PONG     = 0xA;

	const OPCODE_CONTROL_RESERVED_1 = 0xB;
	const OPCODE_CONTROL_RESERVED_2 = 0xC;
	const OPCODE_CONTROL_RESERVED_3 = 0xD;
	const OPCODE_CONTROL_RESERVED_4 = 0xE;
	const OPCODE_CONTROL_RESERVED_5 = 0xF;

	protected $fin = 1; // only one frame is necessary
	protected $rsv = array(0, 0, 0); // rsv1, rsv2, rsv3

	protected $mask    = false;
	protected $maskKey = "\x00\x00\x00\x00";

	protected $opCode;

	/**
	 * Mask a data according to the current mask key
	 *
	 * @param string $data Data to mask
	 * @return string Masked data
	 */
	protected function maskData($data)
	{
		$masked = '';
		$data   = str_split($data);
		$key    = str_split($this->maskKey);

		foreach ($data as $i => $letter) {
			$masked .= $letter ^ $key[$i % 4];
		}

		return $masked;
	}
}

?>