<?php
include "./core/socketcore.php";

/**
 * DPServer Class
 * 
 * @version 0.5
 */
class DPServer {
	private $url = "http://core.dpush.co.kr:9999";	// REAL
	private $productkey = "";
	private $client;
	private $options = array();
	private $result = array();		// ERRORCODE, ERRORDESC 형태의 배열
	private $rcdata = "";			// 실제 받은 데이터 에서 RESULT값만 분리

	/**
	 * DPServer
	 * 
	 * @param string $productkey 발급받은 PRODUCTKEY
	 * @param array $options 옵션
	 */
	public function __construct($productkey, $options = null) {
		$this->productkey = $productkey;
		if($options == null) {
			$options = array();
		}
		$options["productkey"] = $productkey;
		$this->options = $options;
		
		$this->connect();
		$this->receive();

		$this->client->close();
	}
	
	/**
	 * <pre>
	 * 인증관련 키를 생성한다. 또한 CUSTOMER INFO를 사용하는 경우 해당 user에 CUSTOMER INFO를 전달한다.
	 * 
	 * $message = new DPMessage('TESTPRODUCTKEY');
	 * $messge->getAuthKey('chat-group', '7VwIPeMe3jmcrwB5AAAA', array('nickname' => 'Benjamin', 'age' => '25', 'gender' => 'M'));
	 * </pre>
	 * 
	 * @param string $groupname 임의의 그룹명
	 * @param string $userid userid
	 * @param array $custinfo 접속 성공할 경우 CUSTOMER INFO
	 * @returns string $authkey 인증키
	 */
	function genAuthKey($groupname, $userid, $custinfo = array()) {
		$this->connect();
		$this->receive();

		$authkey = "";

		if($this->result['ERRORCODE'] == 0) {
			$this->client->emit('7004', array('GN' => $groupname, 'SI' => $userid, 'CI' => json_encode($custinfo)));
			$_result = $this->client->read();		// 결과에 대한 응답 (3009)
			$_tmparr = split(",",$_result,2);
			$jsondata = json_decode($_tmparr[1]);
			$rcdata = json_decode($jsondata[1],true);

			if($rcdata['RC'] == 0) {
				$authkey = "{\"authkey\":\"" . $rcdata['AK'] . "\"}";
			} else {
				$authkey = "{\"authkey\":\"\"}";
			}
		}
		$this->client->close();
		 
		return $authkey;
	}


	/**
	 * CUSTOMER INFO를 사용하는 경우 해당 user에 변경된 CUSTOMER INFO를 전달한다.
	 * 
	 * $message = new DPMessage('TESTPRODUCTKEY');
	 * $message->updateUser('chat-group', '7VwIPeMe3jmcrwB5AAAA', array('nickname' => 'Benjamin', 'age' => '25', 'gender' => 'M'));
	 * 
	 * @param string $groupname 임의의 그룹명
	 * @param string $userid userid
	 * @param array $custinfo 접속 성공할 경우 CUSTOMER INFO
	 */
	function updateUser($groupname, $userid, $custinfo) {
		$this->connect();
		$this->receive();

		$authkey = "";

		if($this->result['ERRORCODE'] == 0) {
			$this->client->emit('7005', array('GN' => $groupname, 'SI' => $userid, 'CI' => json_encode($custinfo)));
		}
		$this->client->close();
	}
	
	/**
	 * <pre>
	 * actionname으로 Push Data를 전체 Client에게 전달한다.  
	 * opengroup 시에 sendevent 가 true로 설정되어야 동작한다.
	 * 
	 * $message = new DPMessage('TESTPRODUCTKEY');
	 * $message->send('chat-group','chat-action-notice',"Hello World!");
	 * </pre>
	 *  
	 * @param string $groupname 임의의 그룹명
	 * @param string $actionname 임의의 액션명
	 * @param string $message 전달할 데이터의 내용
	 */
	public function send($groupname, $actionname, $message) {
		 
		$this->connect();
		$this->receive();

		if($this->result['ERRORCODE'] == 0) {
			$this->client->emit('7001', array('GN' => $groupname, 'AN' => $actionname, 'MG' => $message));
			$this->receive();
		}

		$this->client->close();
	}
	
	/**
	 * <pre>
	 * 해당 PRODUCTKEY에 등록된 GROUP 리스트를 반환한다. 그룹명을 키로 배열정보를 리턴
	 * 
	 * $message = new DPMessage('TESTPRODUCTKEY');
	 * $groupList = $message->getGroupList();
	 * </pre>
	 * 
	 * @returns array $result 그룹정보 : ['그룹명']=>{['usercnt']:접속유저수, ['custevent']:CUSTOMERINFO 사용유무, ['sendevent']:CLIENT SEND 사용 유무 }
	 */
	public function getGroupList() {
		$result = array();
		 
		$this->connect();
		$this->receive();

		if($this->result['ERRORCODE'] == 0) {
			$this->client->emit('7002', array());
			$this->receive();
			$result = $this->rcdata['GL'];
		}

		$this->client->close();
		 
		return $result;
	}
	
	/**
	 * <pre>
	 * 해당 GROUP에 등록된  유저 리스트를 반환한다.
	 *  
	 * $message = new DPMessage('TESTPRODUCTKEY');
	 * $userList = $message->getUserList('chat-group');
	 * </pre>
	 * 
	 * @param string $groupname 임의의 그룹명
	 * @returns array $result 유저정보 : $userlist['test-chat'] -> 각 Client의 CUSTOMER INFO 배열리턴
	 */
	public function getUserList($groupname) {
		$result = array();
		 
		$this->connect();
		$this->receive();

		if($this->result['ERRORCODE'] == 0) {
			$this->client->emit('7003', array('GN' => $groupname));
			$this->receive();
			$result = $this->rcdata['UL'];
		}

		$this->client->close();
		 
		return $result;
	}

	private function connect() {
		$this->client = new Client(new Version1X($this->url, $this->productkey, $this->options));
		$this->client->initialize();			// 초기화
	}

	private function receive() {
		$_result = $this->client->read();		// 결과에 대한 응답
		$this->resultParser($_result);		// 응답결과
	}

	private function resultParser($result) {
		$_tmparr = explode(",",$result,2);
		$jsondata = json_decode($_tmparr[1]);
		$rcdata = json_decode($jsondata[1],true);
		$this->rcdata = $rcdata;				// 결과값을 처리하기 위해
		$this->result = array (
				'ERRORCODE' => $rcdata['RC'],
				'ERRORDESC' => ''
		);
	}
	
	/**
	 * <pre>
	 * 마지막 함수의 통신 결과를 보여준다. Debug용 함수
	 * 
	 * $message = new DPMessage('TESTPRODUCTKEY');
	 * // TODO 특정 함수실행
	 * var_dump($message->getResult());
	 * </pre>
	 * 
	 * @returns array $result 에러코드 array : ERRORCODE {int}  0이면 정상 아니면 에러,	ERRORDESC {string}  정상은 빈 문자열 에러일 경우 에러 내용
	 */
	public function getResult() {
		return $this->result;
	}
}
